
#include "socket_examples.h"

#include "lwip/opt.h"

#if LWIP_SOCKET

#include "lwip/sockets.h"
#include "lwip/sys.h"

#include "lwipcfg_msvc.h"
#include "lwip/memp.h"
#include "lwip/tcp_impl.h"

#if LWIP_SO_LINGER
/* for extra test purposes only: */
#include "lwip/netif.h"
#endif /* LWIP_SO_LINGER */

#include <string.h>
#include <stdio.h>

#ifndef SOCK_TARGET_HOST
#define SOCK_TARGET_HOST  "192.168.1.1"
#endif

#ifndef SOCK_TARGET_PORT
#define SOCK_TARGET_PORT  80
#endif

#ifndef SOCK_TARGET_MAXHTTPPAGESIZE
#define SOCK_TARGET_MAXHTTPPAGESIZE 1024
#endif

#ifndef SOCKET_EXAMPLES_RUN_PARALLEL
#define SOCKET_EXAMPLES_RUN_PARALLEL 0
#endif

const u8_t cmpbuf[8] = {0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab};

/* a helper struct to ensure memory before/after fd_set is not touched */
typedef struct _xx
{
  u8_t buf1[8];
  fd_set readset;
  u8_t buf2[8];
  fd_set writeset;
  u8_t buf3[8];
  fd_set errset;
  u8_t buf4[8];
} fdsets;

#define INIT_FDSETS(sets) do { \
  memset((sets)->buf1, 0xab, 8); \
  memset((sets)->buf2, 0xab, 8); \
  memset((sets)->buf3, 0xab, 8); \
  memset((sets)->buf4, 0xab, 8); \
}while(0)

#define CHECK_FDSETS(sets) do { \
  LWIP_ASSERT("buf1 fail", !memcmp((sets)->buf1, cmpbuf, 8)); \
  LWIP_ASSERT("buf2 fail", !memcmp((sets)->buf2, cmpbuf, 8)); \
  LWIP_ASSERT("buf3 fail", !memcmp((sets)->buf3, cmpbuf, 8)); \
  LWIP_ASSERT("buf4 fail", !memcmp((sets)->buf4, cmpbuf, 8)); \
}while(0)


/** This is an example function that tests
    blocking- and nonblocking connect. */
static void
sockex_nonblocking_connect(void *arg)
{
  int s;
  int ret;
  struct sockaddr_in addr;
  fdsets sets;
#if 0
  u32_t opt;
  struct timeval tv;
  u32_t ticks_a, ticks_b;
  int err;
#endif
  INIT_FDSETS(&sets);

  LWIP_UNUSED_ARG(arg);
  /* set up address to connect to */
  memset(&addr, 0, sizeof(addr));
  addr.sin_len = sizeof(addr);
  addr.sin_family = AF_INET;
  addr.sin_port = PP_HTONS(SOCK_TARGET_PORT);
  addr.sin_addr.s_addr = inet_addr(SOCK_TARGET_HOST);

  /* first try blocking: */

  /* create the socket */
  printf("- lwip_socket\n");
  s = lwip_socket(AF_INET, SOCK_STREAM, 0);
  printf("- done\n");
  LWIP_ASSERT("s >= 0", s >= 0);
  LWIP_ASSERT("s == 0", s == 0);

  /* connect */
  printf("-lwip_lwip_connectsocket\n");
  ret = lwip_connect(s, (struct sockaddr*)&addr, sizeof(addr));
  printf("- done\n");
  /* should succeed */
  LWIP_ASSERT("ret == 0", ret == 0);

  /* write something */
  printf("-lwip_write\n");
  ret = lwip_write(s, "test", 4);
  printf("- done\n");
  LWIP_ASSERT("ret == 4", ret == 4);

  /* close */
  printf("-lwip_close (%lu)\n", sys_now());
  ret = lwip_close(s);
  printf("- done\n");
  LWIP_ASSERT("ret == 0", ret == 0);
#if 0
  /* now try nonblocking and close before being connected */

  /* create the socket */
  printf("-lwip_socket\n");
  s = lwip_socket(AF_INET, SOCK_STREAM, 0);
  printf("- done\n");
  LWIP_ASSERT("s >= 0", s >= 0);
  LWIP_ASSERT("s == 0", s == 0);
  /* nonblocking */
  printf("-lwip_fcntl\n");
  opt = lwip_fcntl(s, F_GETFL, 0);
  printf("- done\n");
  LWIP_ASSERT("opt != -1", opt != -1);
  opt |= O_NONBLOCK;
  printf("-lwip_fcntl\n");
  ret = lwip_fcntl(s, F_SETFL, opt);
  printf("- done\n");
  LWIP_ASSERT("ret != -1", ret != -1);
  /* connect */
  printf("-lwip_connect\n");
  ret = lwip_connect(s, (struct sockaddr*)&addr, sizeof(addr));
  printf("- done\n");
  /* should have an error: "inprogress" */
  LWIP_ASSERT("ret == -1", ret == -1);
  err = errno;
  LWIP_ASSERT("errno == EINPROGRESS", err == EINPROGRESS);
  /* close */
  ret = lwip_close(s);
  LWIP_ASSERT("ret == 0", ret == 0);
  /* try to close again, should fail with EBADF */
  ret = lwip_close(s);
  LWIP_ASSERT("ret == -1", ret == -1);
  err = errno;
  LWIP_ASSERT("errno == EBADF", err == EBADF);
  printf("closing socket in nonblocking connect succeeded\n");

  /* now try nonblocking, connect should succeed:
     this test only works if it is fast enough, i.e. no breakpoints, please! */

  /* create the socket */
  s = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("s >= 0", s >= 0);
  LWIP_ASSERT("s == 0", s == 0);

  /* nonblocking */
  opt = 1;
  ret = lwip_ioctl(s, FIONBIO, &opt);
  LWIP_ASSERT("ret == 0", ret == 0);

  /* connect */
  ret = lwip_connect(s, (struct sockaddr*)&addr, sizeof(addr));
  /* should have an error: "inprogress" */
  LWIP_ASSERT("ret == -1", ret == -1);
  err = errno;
  LWIP_ASSERT("errno == EINPROGRESS", err == EINPROGRESS);

  /* write should fail, too */
  ret = lwip_write(s, "test", 4);
  LWIP_ASSERT("ret == -1", ret == -1);
  err = errno;
  LWIP_ASSERT("errno == EINPROGRESS", err == EINPROGRESS);

  CHECK_FDSETS(&sets);
  FD_ZERO(&sets.readset);
  CHECK_FDSETS(&sets);
  FD_SET(s, &sets.readset);
  CHECK_FDSETS(&sets);
  FD_ZERO(&sets.writeset);
  CHECK_FDSETS(&sets);
  FD_SET(s, &sets.writeset);
  CHECK_FDSETS(&sets);
  FD_ZERO(&sets.errset);
  CHECK_FDSETS(&sets);
  FD_SET(s, &sets.errset);
  CHECK_FDSETS(&sets);
  tv.tv_sec = 0;
  tv.tv_usec = 0;
  /* select without waiting should fail */
  ret = lwip_select(s + 1, &sets.readset, &sets.writeset, &sets.errset, &tv);
  CHECK_FDSETS(&sets);
  LWIP_ASSERT("ret == 0", ret == 0);
  LWIP_ASSERT("!FD_ISSET(s, &writeset)", !FD_ISSET(s, &sets.writeset));
  LWIP_ASSERT("!FD_ISSET(s, &readset)", !FD_ISSET(s, &sets.readset));
  LWIP_ASSERT("!FD_ISSET(s, &errset)", !FD_ISSET(s, &sets.errset));

  FD_ZERO(&sets.readset);
  FD_SET(s, &sets.readset);
  FD_ZERO(&sets.writeset);
  FD_SET(s, &sets.writeset);
  FD_ZERO(&sets.errset);
  FD_SET(s, &sets.errset);
  ticks_a = sys_now();
  /* select with waiting should succeed */
  ret = lwip_select(s + 1, &sets.readset, &sets.writeset, &sets.errset, NULL);
  ticks_b = sys_now();
  LWIP_ASSERT("ret == 1", ret == 1);
  LWIP_ASSERT("FD_ISSET(s, &writeset)", FD_ISSET(s, &sets.writeset));
  LWIP_ASSERT("!FD_ISSET(s, &readset)", !FD_ISSET(s, &sets.readset));
  LWIP_ASSERT("!FD_ISSET(s, &errset)", !FD_ISSET(s, &sets.errset));

  /* now write should succeed */
  ret = lwip_write(s, "test", 4);
  LWIP_ASSERT("ret == 4", ret == 4);

  /* close */
  ret = lwip_close(s);
  LWIP_ASSERT("ret == 0", ret == 0);

  printf("select() needed %d ticks to return writable\n", ticks_b - ticks_a);


  /* now try nonblocking to invalid address:
     this test only works if it is fast enough, i.e. no breakpoints, please! */

  /* create the socket */
  s = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("s >= 0", s >= 0);
  LWIP_ASSERT("s == 0", s == 0);

  /* nonblocking */
  opt = 1;
  ret = lwip_ioctl(s, FIONBIO, &opt);
  LWIP_ASSERT("ret == 0", ret == 0);

  addr.sin_addr.s_addr++; /* this should result in an invalid address */

  /* connect */
  ret = lwip_connect(s, (struct sockaddr*)&addr, sizeof(addr));
  /* should have an error: "inprogress" */
  LWIP_ASSERT("ret == -1", ret == -1);
  err = errno;
  LWIP_ASSERT("errno == EINPROGRESS", err == EINPROGRESS);

  /* write should fail, too */
  ret = lwip_write(s, "test", 4);
  LWIP_ASSERT("ret == -1", ret == -1);
  err = errno;
  LWIP_ASSERT("errno == EINPROGRESS", err == EINPROGRESS);

  FD_ZERO(&sets.readset);
  FD_SET(s, &sets.readset);
  FD_ZERO(&sets.writeset);
  FD_SET(s, &sets.writeset);
  FD_ZERO(&sets.errset);
  FD_SET(s, &sets.errset);
  tv.tv_sec = 0;
  tv.tv_usec = 0;
  /* select without waiting should fail */
  ret = lwip_select(s + 1, &sets.readset, &sets.writeset, &sets.errset, &tv);
  LWIP_ASSERT("ret == 0", ret == 0);

  FD_ZERO(&sets.readset);
  FD_SET(s, &sets.readset);
  FD_ZERO(&sets.writeset);
  FD_SET(s, &sets.writeset);
  FD_ZERO(&sets.errset);
  FD_SET(s, &sets.errset);
  ticks_a = sys_now();
  /* select with waiting should eventually succeed and return errset! */
  ret = lwip_select(s + 1, &sets.readset, &sets.writeset, &sets.errset, NULL);
  ticks_b = sys_now();
  LWIP_ASSERT("ret > 0", ret > 0);
  LWIP_ASSERT("FD_ISSET(s, &errset)", FD_ISSET(s, &sets.errset));
  /* we do set read/write, too, to ensure everyone gets unblocked */

  /* close */
  printf("-close\n");
  ret = lwip_close(s);
  printf("-close done\n");
  LWIP_ASSERT("ret == 0", ret == 0);

  printf("select() needed %d ticks to return error\n", ticks_b - ticks_a);
#endif
  printf("all tests done, thread ending\n");
}

/** This is an example function that tests
    the recv function (timeout etc.). */
static void
sockex_testrecv(void *arg)
{
  int s;
  int ret;
  int err;
  int opt, opt2;
  socklen_t opt2size;
  struct sockaddr_in addr;
  size_t len;
  char rxbuf[SOCK_TARGET_MAXHTTPPAGESIZE];
  fd_set readset;
  fd_set errset;
  struct timeval tv;

  LWIP_UNUSED_ARG(arg);
  /* set up address to connect to */
  memset(&addr, 0, sizeof(addr));
  addr.sin_len = sizeof(addr);
  addr.sin_family = AF_INET;
  addr.sin_port = PP_HTONS(SOCK_TARGET_PORT);
  addr.sin_addr.s_addr = inet_addr(SOCK_TARGET_HOST);

  /* create the socket */
  s = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("s >= 0", s >= 0);
  LWIP_ASSERT("s == 0", s == 0);

  /* connect */
  ret = lwip_connect(s, (struct sockaddr*)&addr, sizeof(addr));
  /* should succeed */
  LWIP_ASSERT("ret == 0", ret == 0);

  /* set recv timeout (100 ms) */
  opt = 100;
  ret = lwip_setsockopt(s, SOL_SOCKET, SO_RCVTIMEO, &opt, sizeof(int));
  LWIP_ASSERT("ret == 0", ret == 0);
  opt2 = 0;
  opt2size = sizeof(opt2);
  ret = lwip_getsockopt(s, SOL_SOCKET, SO_RCVTIMEO, &opt2, &opt2size);
  LWIP_ASSERT("ret == 0", ret == 0);
  LWIP_ASSERT("opt2size == sizeof(opt2)", opt2size == sizeof(opt2));
  LWIP_ASSERT("opt == opt2", opt == opt2);

  /* write the start of a GET request */
#define SNDSTR1 "G"
  len = strlen(SNDSTR1);
  ret = lwip_write(s, SNDSTR1, len);
  LWIP_ASSERT("ret == len", ret == (int)len);

  /* should time out if the other side is a good HTTP server */
  ret = lwip_read(s, rxbuf, 1);
  LWIP_ASSERT("ret == -1", ret == -1);
  err = errno;
  LWIP_ASSERT("errno == EAGAIN", err == EAGAIN);

  /* write the rest of a GET request */
#define SNDSTR2 "ET / HTTP_1.1\r\n\r\n"
  len = strlen(SNDSTR2);
  ret = lwip_write(s, SNDSTR2, len);
  LWIP_ASSERT("ret == len", ret == (int)len);

  /* wait a while: should be enough for the server to send a response */
  sys_msleep(1000);

  /* should not time out but receive a response */
  ret = lwip_read(s, rxbuf, SOCK_TARGET_MAXHTTPPAGESIZE);
  LWIP_ASSERT("ret > 0", ret > 0);

  /* now select should directly return because the socket is readable */
  FD_ZERO(&readset);
  FD_ZERO(&errset);
  FD_SET(s, &readset);
  FD_SET(s, &errset);
  tv.tv_sec = 10;
  tv.tv_usec = 0;
  ret = lwip_select(s + 1, &readset, NULL, &errset, &tv);
  LWIP_ASSERT("ret == 1", ret == 1);
  LWIP_ASSERT("!FD_ISSET(s, &errset)", !FD_ISSET(s, &errset));
  LWIP_ASSERT("FD_ISSET(s, &readset)", FD_ISSET(s, &readset));

  /* should not time out but receive a response */
  ret = lwip_read(s, rxbuf, SOCK_TARGET_MAXHTTPPAGESIZE);
  /* might receive a second packet for HTTP/1.1 servers */
  if (ret > 0) {
    /* should return 0: closed */
    ret = lwip_read(s, rxbuf, SOCK_TARGET_MAXHTTPPAGESIZE);
    LWIP_ASSERT("ret == 0", ret == 0);
  }

  /* close */
  ret = lwip_close(s);
  LWIP_ASSERT("ret == 0", ret == 0);

  printf("sockex_testrecv finished successfully\n");
}

/** helper struct for the 2 functions below (multithreaded: thread-argument) */
struct sockex_select_helper {
  int socket;
  int wait_read;
  int expect_read;
  int wait_write;
  int expect_write;
  int wait_err;
  int expect_err;
  int wait_ms;
  sys_sem_t sem;
};

/** helper thread to wait for socket events using select */
static void
sockex_select_waiter(void *arg)
{
  struct sockex_select_helper *helper = (struct sockex_select_helper *)arg;
  int ret;
  fd_set readset;
  fd_set writeset;
  fd_set errset;
  struct timeval tv;

  LWIP_ASSERT("helper != NULL", helper != NULL);

  FD_ZERO(&readset);
  FD_ZERO(&writeset);
  FD_ZERO(&errset);
  if (helper->wait_read) {
    FD_SET(helper->socket, &readset);
  }
  if (helper->wait_write) {
    FD_SET(helper->socket, &writeset);
  }
  if (helper->wait_err) {
    FD_SET(helper->socket, &errset);
  }

  tv.tv_sec = helper->wait_ms / 1000;
  tv.tv_usec = (helper->wait_ms % 1000) * 1000;

  ret = lwip_select(helper->socket, &readset, &writeset, &errset, &tv);
  if (helper->expect_read || helper->expect_write || helper->expect_err) {
    LWIP_ASSERT("ret > 0", ret > 0);
  } else {
    LWIP_ASSERT("ret == 0", ret == 0);
  }
  if (helper->expect_read) {
    LWIP_ASSERT("FD_ISSET(helper->socket, &readset)", FD_ISSET(helper->socket, &readset));
  } else {
    LWIP_ASSERT("!FD_ISSET(helper->socket, &readset)", !FD_ISSET(helper->socket, &readset));
  }
  if (helper->expect_write) {
    LWIP_ASSERT("FD_ISSET(helper->socket, &writeset)", FD_ISSET(helper->socket, &writeset));
  } else {
    LWIP_ASSERT("!FD_ISSET(helper->socket, &writeset)", !FD_ISSET(helper->socket, &writeset));
  }
  if (helper->expect_err) {
    LWIP_ASSERT("FD_ISSET(helper->socket, &errset)", FD_ISSET(helper->socket, &errset));
  } else {
    LWIP_ASSERT("!FD_ISSET(helper->socket, &errset)", !FD_ISSET(helper->socket, &errset));
  }
  sys_sem_signal(&helper->sem);
}

/** This is an example function that tests
    more than one thread being active in select. */
static void
sockex_testtwoselects(void *arg)
{
  int s1;
  int s2;
  int ret;
  struct sockaddr_in addr;
  size_t len;
  err_t lwiperr;
  struct sockex_select_helper h1, h2, h3, h4;

  LWIP_UNUSED_ARG(arg);
  /* set up address to connect to */
  memset(&addr, 0, sizeof(addr));
  addr.sin_len = sizeof(addr);
  addr.sin_family = AF_INET;
  addr.sin_port = PP_HTONS(SOCK_TARGET_PORT);
  addr.sin_addr.s_addr = inet_addr(SOCK_TARGET_HOST);

  /* create the sockets */
  s1 = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("s1 >= 0", s1 >= 0);
  LWIP_ASSERT("s1 == 0", s1 == 0);
  s2 = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("s2 >= 0", s2 >= 0);

  /* connect, should succeed */
  ret = lwip_connect(s1, (struct sockaddr*)&addr, sizeof(addr));
  LWIP_ASSERT("ret == 0", ret == 0);
  ret = lwip_connect(s2, (struct sockaddr*)&addr, sizeof(addr));
  LWIP_ASSERT("ret == 0", ret == 0);

  /* write the start of a GET request */
#define SNDSTR1 "G"
  len = strlen(SNDSTR1);
  ret = lwip_write(s1, SNDSTR1, len);
  LWIP_ASSERT("ret == len", ret == (int)len);
  ret = lwip_write(s2, SNDSTR1, len);
  LWIP_ASSERT("ret == len", ret == (int)len);

  h1.wait_read  = 1;
  h1.wait_write = 1;
  h1.wait_err   = 1;
  h1.expect_read  = 0;
  h1.expect_write = 0;
  h1.expect_err   = 0;
  lwiperr = sys_sem_new(&h1.sem, 0);
  LWIP_ASSERT("lwiperr == ERR_OK", lwiperr == ERR_OK);
  h1.socket = s1;
  h1.wait_ms = 500;

  h2 = h1;
  lwiperr = sys_sem_new(&h2.sem, 0);
  LWIP_ASSERT("lwiperr == ERR_OK", lwiperr == ERR_OK);
  h2.socket = s2;
  h2.wait_ms = 1000;

  h3 = h1;
  lwiperr = sys_sem_new(&h3.sem, 0);
  LWIP_ASSERT("lwiperr == ERR_OK", lwiperr == ERR_OK);
  h3.socket = s2;
  h3.wait_ms = 1500;

  h4 = h1;
  lwiperr = sys_sem_new(&h4.sem, 0);
  LWIP_ASSERT("lwiperr == ERR_OK", lwiperr == ERR_OK);
  h4.socket = s2;
  h4.wait_ms = 2000;

  /* select: all sockets should time out if the other side is a good HTTP server */

  sys_thread_new("sockex_select_waiter1", sockex_select_waiter, &h2, 0, 0);
  sys_msleep(100);
  sys_thread_new("sockex_select_waiter2", sockex_select_waiter, &h1, 0, 0);
  sys_msleep(100);
  sys_thread_new("sockex_select_waiter2", sockex_select_waiter, &h4, 0, 0);
  sys_msleep(100);
  sys_thread_new("sockex_select_waiter2", sockex_select_waiter, &h3, 0, 0);

  sys_sem_wait(&h1.sem);
  sys_sem_wait(&h2.sem);
  sys_sem_wait(&h3.sem);
  sys_sem_wait(&h4.sem);

  /* close */
  ret = lwip_close(s1);
  LWIP_ASSERT("ret == 0", ret == 0);
  ret = lwip_close(s2);
  LWIP_ASSERT("ret == 0", ret == 0);

  printf("sockex_testtwoselects finished successfully\n");
}

static netif_input_fn original_input_fn;
u32_t ignore_ipv4;

static err_t
sockex_selective_dropping_netif_input(struct pbuf *p, struct netif *inp)
{
  u16_t proto;
  u32_t src_ip;

  if (p != NULL) {
    if (pbuf_copy_partial(p, &proto, 2, 12) == 2) {
      if (proto == PP_HTONS(0x0800)) { /* IPv4 */
        if (pbuf_copy_partial(p, &src_ip, 4, 26)) {
          if (src_ip == ignore_ipv4) {
            pbuf_free(p);
            return ERR_OK;
          }
        }
      }
    }
  }
  if (original_input_fn != NULL) {
    return original_input_fn(p, inp);
  }
  if (p != NULL) {
    pbuf_free(p);
  }
  return ERR_OK;
}

/* This ugly operation hacks the input function of the default netif to
 * ensure we don't receiveany more packets from the selected host.
 * DON'T TRY THIS AT HOME!
 */
static void
sockex_ignorethisip(ip4_addr_t* ip_to_ignore)
{
  LWIP_ASSERT("no default netif", netif_default != NULL);
  ignore_ipv4 = ip_to_ignore->addr;
  original_input_fn = netif_default->input;
  netif_default->input = sockex_selective_dropping_netif_input;
}

static void
sockex_input_back_to_default(void)
{
  LWIP_ASSERT("no default netif", netif_default != NULL);
  ignore_ipv4 = 0;
  netif_default->input = original_input_fn;
}

/** This is an example function that tests
    the linger function */
static void
sockex_testlinger_block_nonblock(u8_t nonblocking, int linger, int lingertime_sec, int forcememerr)
{
  int s;
  int ret;
  int err;
  int opt;
  struct sockaddr_in addr;
  char rxbuf[SOCK_TARGET_MAXHTTPPAGESIZE];
  fd_set writeset;
  fd_set readset;
  fd_set errset;
  u32_t ticks_a, ticks_b;
  ip4_addr_t remote;
  size_t write_size;
  void* mems = NULL;
#if LWIP_SO_LINGER
  struct linger ling, ling2;
  socklen_t opt2size;
#else
  LWIP_UNUSED_ARG(linger);
  LWIP_UNUSED_ARG(lingertime_sec);
#endif

  /* set up address to connect to */
  memset(&addr, 0, sizeof(addr));
  addr.sin_len = sizeof(addr);
  addr.sin_family = AF_INET;
  addr.sin_port = PP_HTONS(SOCK_TARGET_PORT);
  addr.sin_addr.s_addr = inet_addr(SOCK_TARGET_HOST);

  /* create the socket */
  s = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("s >= 0", s >= 0);
  LWIP_ASSERT("s == 0", s == 0);

  if (nonblocking) {
    /* nonblocking */
    opt = lwip_fcntl(s, F_GETFL, 0);
    LWIP_ASSERT("opt != -1", opt != -1);
    opt |= O_NONBLOCK;
    ret = lwip_fcntl(s, F_SETFL, opt);
    LWIP_ASSERT("ret != -1", ret != -1);
  }

  /* connect */
  ret = lwip_connect(s, (struct sockaddr*)&addr, sizeof(addr));
  /* should succeed */
  if (nonblocking) {
    LWIP_ASSERT("ret == -1", ret == -1);
    err = errno;
    LWIP_ASSERT("errno == EINPROGRESS", err == EINPROGRESS);
    FD_ZERO(&writeset);
    FD_SET(s, &writeset);
    FD_ZERO(&readset);
    FD_SET(s, &readset);
    FD_ZERO(&errset);
    FD_SET(s, &errset);
    ticks_a = sys_now();
    /* select with waiting should succeed */
    ret = lwip_select(s + 1, &readset, &writeset, &errset, NULL);
    ticks_b = sys_now();
    printf("select needed %d ms\n", ticks_b - ticks_a);
    LWIP_ASSERT("ret == 1", ret == 1);
    LWIP_ASSERT("FD_ISSET(s, &writeset)", FD_ISSET(s, &writeset));
    LWIP_ASSERT("!FD_ISSET(s, &readset)", !FD_ISSET(s, &readset));
    LWIP_ASSERT("!FD_ISSET(s, &errset)", !FD_ISSET(s, &errset));
  } else {
    LWIP_ASSERT("ret == 0", ret == 0);
  }

#if LWIP_SO_LINGER
  ling.l_onoff = linger;
  ling.l_linger = lingertime_sec;
  ret = lwip_setsockopt(s, SOL_SOCKET, SO_LINGER, &ling, sizeof(ling));
  LWIP_ASSERT("ret == 0", ret == 0);
  opt2size = sizeof(ling);
  ret = lwip_getsockopt(s, SOL_SOCKET, SO_LINGER, &ling2, &opt2size);
  LWIP_ASSERT("ret == 0", ret == 0);
  LWIP_ASSERT("ling == ling2", ling.l_onoff == ling2.l_onoff && ling.l_linger == ling2.l_linger);
#endif

  /* prevent getting any more packets from remote host */
  remote.addr = ipaddr_addr(SOCK_TARGET_HOST);
  sockex_ignorethisip(&remote);

  /* write many bytes */
  write_size = forcememerr ? TCP_MSS : TCP_MSS * 4;
  ret = lwip_write(s, rxbuf, write_size);
  LWIP_ASSERT("ret > 0", ret > 0);

  if (forcememerr) {
    /* use all tcp_seg structs to force memory error */
    void *last = NULL, *cur = NULL;
    do {
      cur = mem_malloc(4);
      if (cur != NULL) {
        *(void**)cur = NULL;
        if (last == NULL) {
          last = cur;
          mems = cur;
        } else {
          *(void**)last = cur;
          last = cur;
        }
      }
    } while(cur != NULL);
  }

  ticks_a = sys_now();
  /* close */
  ret = lwip_close(s);
  ticks_b = sys_now();
  if(ret != 0)
  {
    printf("close failed with error %d after %d ms\n", errno, ticks_b - ticks_a);
#if LWIP_SO_LINGER
    /* disable linger and close again */
    ling.l_onoff = 0;
    ling.l_linger = 0;
    ret = lwip_setsockopt(s, SOL_SOCKET, SO_LINGER, &ling, sizeof(ling));
    LWIP_ASSERT("ret == 0", ret == 0);
    ret = lwip_close(s);
    LWIP_ASSERT("ret == 0", ret == 0);
#endif
  }
  else
  {
    LWIP_ASSERT("ret == 0", ret == 0);
    printf("close needed %d ms\n", ticks_b - ticks_a);
  }
  while (mems != NULL) {
    /* free the unused memory */
    void* f = mems;
    mems = *(void**)mems;
    mem_free(f);
  }
  sockex_input_back_to_default();
}

/** This is an example function that tests
    the linger function */
static void
sockex_udp_block_nonblock(u8_t nonblocking)
{
  int s;
  int ret;
  int opt;
  struct sockaddr_in addr;
  char rxbuf[TCP_MSS];
  fd_set writeset;
  fd_set readset;
  fd_set errset;
  u32_t ticks_a, ticks_b;
  size_t write_size;

  /* set up address to connect to */
  memset(&addr, 0, sizeof(addr));
  addr.sin_len = sizeof(addr);
  addr.sin_family = AF_INET;
  addr.sin_port = PP_HTONS(53);
  addr.sin_addr.s_addr = inet_addr("8.8.8.8");

  /* create the socket */
  s = lwip_socket(AF_INET, SOCK_DGRAM, 0);
  LWIP_ASSERT("s >= 0", s >= 0);
  LWIP_ASSERT("s == 0", s == 0);

  if (nonblocking) {
    /* nonblocking */
    opt = lwip_fcntl(s, F_GETFL, 0);
    LWIP_ASSERT("opt != -1", opt != -1);
    opt |= O_NONBLOCK;
    ret = lwip_fcntl(s, F_SETFL, opt);
    LWIP_ASSERT("ret != -1", ret != -1);
  }

  /* connect */
  ret = lwip_connect(s, (struct sockaddr*)&addr, sizeof(addr));
  /* should succeed */
  if (nonblocking) {
    FD_ZERO(&writeset);
    FD_SET(s, &writeset);
    FD_ZERO(&readset);
    FD_SET(s, &readset);
    FD_ZERO(&errset);
    FD_SET(s, &errset);
    ticks_a = sys_now();
    /* select with waiting should succeed */
    ret = lwip_select(s + 1, &readset, &writeset, &errset, NULL);
    ticks_b = sys_now();
    printf("select needed %d ms\n", ticks_b - ticks_a);
    LWIP_ASSERT("ret == 1", ret == 1);
    LWIP_ASSERT("FD_ISSET(s, &writeset)", FD_ISSET(s, &writeset));
    LWIP_ASSERT("!FD_ISSET(s, &readset)", !FD_ISSET(s, &readset));
    LWIP_ASSERT("!FD_ISSET(s, &errset)", !FD_ISSET(s, &errset));
  } else {
    LWIP_ASSERT("ret == 0", ret == 0);
  }

  /* write one request */
  write_size = TCP_MSS;
  ret = lwip_write(s, rxbuf, write_size);
  LWIP_ASSERT("ret > 0", ret > 0);

  ticks_a = sys_now();
  /* close */
  ret = lwip_close(s);
  ticks_b = sys_now();
  if(ret != 0)
  {
    printf("close failed with error %d after %d ms\n", errno, ticks_b - ticks_a);
  }
  else
  {
    LWIP_ASSERT("ret == 0", ret == 0);
    printf("close needed %d ms\n", ticks_b - ticks_a);
  }
}

static void
sockex_testclose(void *arg)
{
  LWIP_UNUSED_ARG(arg);
  sockex_udp_block_nonblock(0);
  sockex_udp_block_nonblock(1);
  sockex_udp_block_nonblock(0);
  sockex_udp_block_nonblock(1);
#if LWIP_SO_LINGER
  sockex_testlinger_block_nonblock(0, 1, 5, 0);
  sockex_testlinger_block_nonblock(0, 1, 0, 0);
  sockex_testlinger_block_nonblock(0, 0, 0, 0);
  sockex_testlinger_block_nonblock(1, 1, 5, 0);
  sockex_testlinger_block_nonblock(1, 1, 0, 0);
  sockex_testlinger_block_nonblock(1, 0, 0, 0);
#endif /* LWIP_SO_LINGER */
  /* force memory error before close to check RST timeout on close without linger */
  sockex_testlinger_block_nonblock(0, 0, 0, 1);

  printf("sockex_testlinger finished successfully\n");
}

struct lwip_sock {
  struct netconn *conn;
  void *lastdata;
  u16_t lastoffset;
  s16_t rcvevent;
  u16_t sendevent;
  u16_t errevent;
  u8_t err;
  u8_t select_waiting;
};
extern struct lwip_sock sockets[MEMP_NUM_NETCONN];


static void
sockex_test_passive_close(void* arg)
{
  int l, s, ret;
  struct sockaddr_in addr;
  struct sockaddr_in remote;
  socklen_t size;
  int len, lastlen;
  u8_t buf[TCP_MSS];

  LWIP_UNUSED_ARG(arg);

  l = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("", l >= 0);
  LWIP_ASSERT("ls == 0", l == 0);

  memset(&addr, 0, sizeof(addr));
  addr.sin_len = sizeof(addr);
  addr.sin_family = AF_INET;
  addr.sin_port = PP_HTONS(25);
  addr.sin_addr.s_addr = INADDR_ANY;
  ret = lwip_bind(l, (struct sockaddr*)&addr, sizeof(addr));
  LWIP_ASSERT("", ret == 0);

  ret = lwip_listen(l, 0);
  LWIP_ASSERT("", ret == 0);
  do {
    memset(&remote, 0, sizeof(remote));
    size = sizeof(remote);
    s = lwip_accept(l, (struct sockaddr*)&remote, &size);
    if (s >= 0) {
      do {
        fd_set fa, fb, fc;
        FD_ZERO(&fa);FD_ZERO(&fb);FD_ZERO(&fc);
        FD_SET(s, &fa);FD_SET(s, &fb);FD_SET(s, &fc);
        ret = lwip_select(s+1, &fa, NULL, &fc, NULL);
        if (ret < 0) {
          return;
        } else if (ret == 0) {
          return;
        } else {
          ret = 0;
        }
        len = lwip_recv(s, &buf, sizeof(buf), 0);
        if (len > 0) {
          lastlen = len;
        } else if (len == 0) {
          ret = lwip_write(s, &buf, len);
          LWIP_ASSERT("", ret == len);
          lwip_close(s);
        } else {
          lwip_close(s);
        }
      } while(len > 0);
    }
  } while (s >= 0);
}

static void
sockex_test_loopback_listen_and_connect(void *arg, int mode)
{
  int s, c, cs;
  int ret;
  int err;
  int opt;
  struct sockaddr_in addr, remote_addr;
  char rxbuf[16];
  socklen_t rlen;
  ip4_addr_t ipaddr;

  LWIP_UNUSED_ARG(arg);

  /* set up address to connect to */
  memset(&addr, 0, sizeof(addr));
  addr.sin_len = sizeof(addr);
  addr.sin_family = AF_INET;
  addr.sin_port = PP_HTONS(25061+mode);
  ipaddr = netif_default->ip_addr;
  ip4_addr_set_u32(&ipaddr, PP_HTONL(IPADDR_LOOPBACK));
  inet_addr_from_ipaddr(&addr.sin_addr, &ipaddr);

  /* create the server socket */
  s = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("s >= 0", s >= 0);
  LWIP_ASSERT("s == 0", s == 0);
  /* create the client socket */
  c = lwip_socket(AF_INET, SOCK_STREAM, 0);
  LWIP_ASSERT("c >= 0", c >= 0);
  LWIP_ASSERT("c == 1", c == 1);

  rlen = sizeof(remote_addr);
  ret = lwip_getpeername(s, (struct sockaddr*)&remote_addr, &rlen);
  LWIP_ASSERT("ret == -1", ret == -1);
  err = errno;
  LWIP_ASSERT("errno == ENOTCONN", err == ENOTCONN);

  /* nonblocking */
  opt = lwip_fcntl(s, F_GETFL, 0);
  LWIP_ASSERT("opt != -1", opt != -1);
  opt |= O_NONBLOCK;
  ret = lwip_fcntl(s, F_SETFL, opt);
  LWIP_ASSERT("ret != -1", ret != -1);
  /*opt = lwip_fcntl(c, F_GETFL, 0);
  LWIP_ASSERT("opt != -1", opt != -1);
  opt |= O_NONBLOCK;
  ret = lwip_fcntl(c, F_SETFL, opt);
  LWIP_ASSERT("ret != -1", ret != -1);*/

  /* bind */
  ret = lwip_bind(s, (struct sockaddr*)&addr, sizeof(addr));
  LWIP_ASSERT("ret == 0", ret == 0);

  /* listen */
  ret = lwip_listen(s, 0);
  LWIP_ASSERT("ret == 0", ret == 0);

  /* connect, nonblocking */
  ret = lwip_connect(c, (struct sockaddr*)&addr, sizeof(addr));
  /* should succeed  */
  LWIP_ASSERT("ret == 0", ret == 0);

  sys_msleep(10);

  /* accept */
  rlen = sizeof(remote_addr);
  cs = lwip_accept(s, (struct sockaddr*)&remote_addr, &rlen);
  LWIP_ASSERT("cs >= 0", cs >= 0);

  if(mode == 0)
  {
    /* client writes something */
    ret = lwip_write(c, "bla", 3);
    LWIP_ASSERT("ret == 3", ret == 3);
    /* server reads */
    ret = lwip_read(cs, rxbuf, sizeof(rxbuf));
    LWIP_ASSERT("ret == 3", ret == 3);
  }
  else if (mode == 1)
  {
    /* clients shuts down its write side */
    ret = lwip_shutdown(c, SHUT_WR);
    LWIP_ASSERT("ret == 0", ret == 0);
  }

  ret = lwip_shutdown(cs, SHUT_RD);
  LWIP_ASSERT("ret == 0", ret == 0);

  ret = lwip_write(cs, "blabla", 6);
  LWIP_ASSERT("ret == 6", ret == 6);
  ret = lwip_shutdown(cs, SHUT_WR);
  LWIP_ASSERT("ret == 0", ret == 0);
  /* -> cs is now fully shut */
  ret = lwip_read(c, rxbuf, sizeof(rxbuf));
  LWIP_ASSERT("ret == 6", ret == 6);

  ret = lwip_close(c);
  LWIP_ASSERT("ret == 0", ret == 0);
  ret = lwip_close(cs);
  LWIP_ASSERT("ret == 0", ret == 0);
  ret = lwip_close(s);
  LWIP_ASSERT("ret == 0", ret == 0);
}

#if !SOCKET_EXAMPLES_RUN_PARALLEL
void socket_example_test(void* arg)
{
  LWIP_UNUSED_ARG(arg);
#if 0
  sockex_nonblocking_connect(NULL);
  sockex_testrecv(NULL);
  sockex_testtwoselects(NULL);
  /*sockex_test_passive_close(NULL);*/
  sockex_testclose(NULL);
#endif
  sockex_test_loopback_listen_and_connect(NULL, 0);
#if 0
  sockex_test_loopback_listen_and_connect(NULL, 1);
#endif
  printf("socket_example_test finished successfully\n");
}
#endif

void socket_examples_init(void)
{
#if SOCKET_EXAMPLES_RUN_PARALLEL
  sys_thread_new("sockex_nonblocking_connect", sockex_nonblocking_connect, NULL, 0, 0);
  sys_thread_new("sockex_testrecv", sockex_testrecv, NULL, 0, 0);
  sys_thread_new("sockex_testtwoselects", sockex_testtwoselects, NULL, 0, 0);
  sys_thread_new("sockex_testclose", sockex_testclose, NULL, 0, 0);
#else
  sys_thread_new("socket_example_test", socket_example_test, NULL, 0, 0);
#endif
}

#endif /* LWIP_SOCKETS */
